#============================================================================== 
# ** Ruby.Higher Classes
#------------------------------------------------------------------------------
# Description:
# ------------
# Methods created to enhance the higher classes in Ruby.
#  
# Method List:
# ------------
#
#   Module
#   ------
#   attr_sec_accessor
#   attr_sec_reader
#   class_accessor
#   class_writer
#   class_reader
#   class_sec_accessor
#   class_sec_reader
#   super_skip
#     
#   Kernel
#   ------
#   class_type
#   rand_between
#   rand_name
#==============================================================================

MACL::Loaded << 'Ruby.Higher Classes'

#============================================================================== 
# ** Module     
#==============================================================================

class Module
  #-------------------------------------------------------------------------
  # * Name      : Secondary Accessor (Reader with a default)
  #   Info      : Creates reader and writer method for one or more 
  #               instance variables, with the reader methods having
  #               default values
  #   Author    : Zeriab
  #   Call Info : Symbol & Default Pairs, and optional Block
  #-------------------------------------------------------------------------
  def attr_sec_accessor(sym, *args, &block)
    if block.nil? # default actions
      args = [0] if args.size == 0 #default = 0
      if args.size < 2 # One pair
        attr_writer sym
        attr_sec_reader sym, *args
      else # loads of methods followed by a default value.
        default = args[-1]
        syms = [sym].concat(args)
        syms.pop
        for sym in syms
          attr_writer sym
          attr_sec_reader sym, default
        end
      end
    else # when a block is given
      # currently just pair sematics
      args.unshift(sym)
      i = 0
      while i < args.size
        attr_writer args[i]
        attr_sec_reader args[i], args[i+1]
        i += 2
      end
    end
  end
  #-------------------------------------------------------------------------
  # * Name      : Secondary Reader (Reader with a default)
  #   Info      : Creates reader method for one instance variable, 
  #               with the reader methods having default a value
  #   Author    : Zeriab
  #   Call Info : Symbol & Default Pair
  #-------------------------------------------------------------------------
  def attr_sec_reader(sym, default = 0)
    sym = sym.id2name
    string = "def #{sym};" +
             "  @#{sym} = #{default}  if @#{sym}.nil?;" +
             "  @#{sym};" +
             "end;"
    module_eval(string)
  end
  #-------------------------------------------------------------------------
  # * Name      : Class Attribute
  #   Info      : Calls class_reader/writer for a symbol
  #   Author    : Yeyinde
  #   Call Info : symbol, writable
  #               symbol:   A Symbol
  #               writable: Boolean telling is symbol is writable
  #-------------------------------------------------------------------------
  private
  def class_attr(symbol, writable = false)
    class_reader(symbol)
    class_writer(symbol) if writable
    return nil
  end
  #-------------------------------------------------------------------------
  # * Name       : Class Accessor
  #   Info       : Creates reader and writer methods for one or more class 
  #                variables
  #   Author     : Yeyinde
  #   Call Info  : One or more Symbols
  #-------------------------------------------------------------------------
  def class_accessor(*symbols)
    class_writer(*symbols)
    class_reader(*symbols)
  end
  #-------------------------------------------------------------------------
  # * Name      : Class Writer
  #   Info      : Creates writer method for one or more class variables
  #   Author    : Yeyinde
  #   Call Info : One or more Symbols
  #-------------------------------------------------------------------------
  def class_writer(*symbols)
    symbols.each do |symbol|
      method = symbol.id2name
      module_eval("def self.#{method}=(var); @@#{method} = var; end")
    end
    return nil
  end
  #-------------------------------------------------------------------------
  # * Name      : Class Reader
  #   Info      : Creates reader method for one or more class variables
  #   Author    : Yeyinde
  #   Call Info : One or more Symbols 
  #-------------------------------------------------------------------------
  def class_reader(*symbols)
    symbols.each do |symbol|
      method = symbol.id2name
      module_eval("def self.#{method}; return @@#{method}; end")
    end
    return nil
  end
  #-------------------------------------------------------------------------
  # * Name      : Class Secondary Accessor (Reader with a default)
  #   Info      : Creates reader and writer method for one or more 
  #               class variables, with the reader methods having
  #               default values
  #   Author    : Zeriab, Yeyinde & SephirothSpawn
  #   Call Info : Symbol & Default Pairs, and optional Block
  #-------------------------------------------------------------------------
  def class_sec_accessor(sym, *args, &block)
    if block.nil? # default actions
      args = [0] if args.size == 0 #default = 0
      if args.size < 2 # One pair
        class_writer sym
        class_sec_reader sym, *args
      else # loads of methods followed by a default value.
        default = args[-1]
        syms = [sym].concat(args)
        syms.pop
        for sym in syms
          class_writer sym
          class_sec_reader sym, default
        end
      end
    else # when a block is given
      # currently just pair sematics
      args.unshift(sym)
      i = 0
      while i < args.size
        class_writer args[i]
        class_sec_reader args[i], args[i+1]
        i += 2
      end
    end
  end
  #-------------------------------------------------------------------------
  # * Name      : Class Secondary Reader (Reader with a default)
  #   Info      : Creates reader method for one class variable, 
  #               with the reader methods having default a value
  #   Author    : Zeriab, Yeyinde & SephirothSpawn
  #   Call Info : Symbol & Default Pair
  #-------------------------------------------------------------------------
  def class_sec_reader(sym, default = 0)
    sym = sym.id2name
    string = "def self.#{sym};" +
             "  @@#{sym} = #{default}  if @@#{sym}.nil?;" +
             "  @@#{sym};" +
             "end;"
    module_eval(string)
  end
  #-------------------------------------------------------------------------
  # * Name      : Super Skip
  #   Info      : Allows you to call ancestor super method, without calling
  #               ancestors between target ancestor.
  #   Author    : SephirothSpawn
  #   Call Info : Method Name (Symbol)
  #               Target Parent Class (Class Name)
  #               Script Name (String)
  #-------------------------------------------------------------------------
  def super_skip(method_name, tpc, script_name = 'modulemethodskip')
    # Gets String Version of Method Name
    method_name = method_name.id2name
    # Gets Array of Ancstors
    ancestors = self.ancestors
    # If Target Parent Class not part of Ancestors
    unless ancestors.include?(tpc)
      p 'Un-defined Parents Class - ' + tpc.to_s
      return
    end
    # Starts Index
    index = 0
    # Gets Target Parent Index
    target_index = ancestors.index(tpc)
    # Loop
    loop do
      # Adds Index
      index += 1
      # Break if Passed Target Index
      break if index >= target_index
      # Gets Class Names
      cname1 = ancestors[index].to_s
      cname2 = ancestors[index - 1].to_s
      # Gets Alias Name
      aliasn = "seph_#{script_name}_#{cname1.downcase}_#{method_name}"
      # Gets String
      string  = "class ::#{cname1};"
      string += "  alias_method :#{aliasn}, :#{method_name};"
      string += "  def #{method_name}(*args);"
      string += "    super(*args);"
      string += "    return if self.is_a?(#{ancestors[index - 1]});"
      string += "    self.#{aliasn}(*args);"
      string += "  end;"
      string += "end;"
      # Evaulates String
      eval string
    end
  end
end

#============================================================================== 
# ** Kernel     
#==============================================================================

module Kernel
  #-------------------------------------------------------------------------
  #   Name      : Class Type
  #   Info      : Returns a string representing the class type
  #   Author    : Trickster
  #   Call Info : No Arguments
  #-------------------------------------------------------------------------
  def class_type
    # Branch By Object's class
    case self
    when FalseClass, TrueClass, NilClass, Numeric, Range, Regexp, String, Symbol
      # Inspection
      string = inspect
    when Array
      # If Object Is Empty
      if empty?
        # Set String
        string = 'Empty Array'
      else
        # Get Class
        data_type = compact[0].class
        # Setup flag
        flag = true
        # Run Through Compacted data checking class
        compact.each {|data| flag &&= data.class == data_type}
        # Setup Extra String to class if flag is true
        string = 'Array - ' + (flag ? data_type.to_s : 'Object')
      end
    when Hash
      # If Object Is Empty
      if empty?
        # Set String
        string = 'Empty Hash'
      else
        # Get Class
        key_type = keys.compact[0].class
        # Setup flag
        key = true
        # Run Through Compacted data checking class
        keys.compact.each {|data| key &&= data.class == key_type}
        # Get Class
        value_type = values.compact[0].class
        # Setup flag
        value = true
        # Run Through Compacted data checking class
        values.compact.each {|data| value &&= data.class == value_type}
        # Setup Extra String to class if flag is true
        string = 'Hash - ' + (key ? key_type.to_s : 'Object') + ' => ' +
          (value ? value_type.to_s : 'Object')
      end
    when Class
      string = self.to_s + ':Class'
    when Module
      string = self.to_s + ':Module'
    else
      # Get Class Name
      string = self.class.to_s
    end
    # Return String
    return string  
  end
  #-------------------------------------------------------------------------
  # * Name      : Random Number Between
  #   Info      : Creates a random number between two numbers
  #   Author    : SephirothSpawn
  #   Call Info : Min and Max Number
  #-------------------------------------------------------------------------
  def rand_between(min, max)
    return min + rand(max - min + 1)
  end
  #-------------------------------------------------------------------------
  # * Name      : Random Name
  #   Info      : Creates a random name
  #   Author    : Trickster
  #   Call Info : Variable - Format Strings If none then strings from Setup
  #               Are Used
  #   Comments  : Format C - consonant V - vowel
  #-------------------------------------------------------------------------
  def rand_name(*formats)
    format = formats.random
    name = ''
    format.split(//).each do |char|
      case char
      when 'C'
        name += %w( B C D F G H J K L M N P R S T V W X Y Z ).random
      when 'V'
        name += %w( A E I O U Y ).random
      when 'c'
        name += %w( b c d f g h j k l m n p r s t v w x y z ).random
      when 'v'
        name += %w( a e i o u y ).random
      else
        name += char
      end
    end
    return name
  end
end